import mongoose from "mongoose";
import dotenv from "dotenv";

dotenv.config();

// Models
const userSchema = new mongoose.Schema({
  username: { type: String, required: true },
  email: { type: String, required: true, unique: true },
  password: { type: String, required: true },
  phoneNumber: { type: String },
  adress: { type: String },
  isUser: { type: Boolean, default: true },
  isAdmin: { type: Boolean, default: false },
  refreshToken: { type: String }
});

const vehicleSchema = new mongoose.Schema({
  registeration_number: { type: String, required: true, unique: true },
  model: { type: String },
  price: { type: Number },
  location: { type: String, required: true },
  district: { type: String, required: true },
  isBooked: { type: Boolean, default: false },
  isAdminApproved: { type: Boolean, default: true }
}, { strict: false });

const bookingSchema = new mongoose.Schema({
  user_id: { type: mongoose.Schema.Types.ObjectId, ref: 'User', required: true },
  vehicle_id: { type: mongoose.Schema.Types.ObjectId, ref: 'Vehicle', required: true },
  totalPrice: { type: Number, required: true },
  pickupDate: { type: String, required: true },
  dropoffDate: { type: String, required: true },
  pickup_district: { type: String, required: true },
  pickup_location: { type: String, required: true },
  dropoff_location: { type: String, required: true },
  razorpayOrderId: { type: String, required: true },
  razorpayPaymentId: { type: String, required: true },
  status: { type: String, default: 'confirmed' },
  createdAt: { type: Date, default: Date.now }
});

const User = mongoose.model("User", userSchema);
const Vehicle = mongoose.model("Vehicle", vehicleSchema);
const Booking = mongoose.model("Booking", bookingSchema);

async function testBookingFlow() {
  console.log("🚗 CAR RENTAL BOOKING TEST SCRIPT");
  console.log("=".repeat(50));
  
  try {
    await mongoose.connect(process.env.mongo_uri);
    console.log("✅ Connected to MongoDB\n");

    // Step 1: Check for test user
    console.log("📋 STEP 1: Checking Test User...");
    let testUser = await User.findOne({ email: "test@gmail.com" });
    if (testUser) {
      console.log(`   ✅ Test user found: ${testUser.username} (${testUser.email})`);
    } else {
      console.log("   ❌ No test user found. Please register first at http://localhost:5173/signup");
      return;
    }

    // Step 2: Check available vehicles
    console.log("\n📋 STEP 2: Checking Available Vehicles...");
    const vehicles = await Vehicle.find({ isAdminApproved: true, isBooked: false });
    console.log(`   ✅ Found ${vehicles.length} available vehicles:`);
    vehicles.forEach((v, i) => {
      console.log(`      ${i + 1}. ${v.model} - ₹${v.price}/day - ${v.district} (${v.location})`);
    });

    if (vehicles.length === 0) {
      console.log("   ❌ No vehicles available. Run seedVehicles.js first.");
      return;
    }

    // Step 3: Create a test booking
    console.log("\n📋 STEP 3: Creating Test Booking...");
    const selectedVehicle = vehicles[0];
    const pickupDate = new Date();
    const dropoffDate = new Date(pickupDate.getTime() + 24 * 60 * 60 * 1000); // +1 day

    const testBooking = new Booking({
      user_id: testUser._id,
      vehicle_id: selectedVehicle._id,
      totalPrice: selectedVehicle.price + 50, // price + shipping
      pickupDate: pickupDate.toISOString(),
      dropoffDate: dropoffDate.toISOString(),
      pickup_district: selectedVehicle.district,
      pickup_location: selectedVehicle.location,
      dropoff_location: selectedVehicle.location,
      razorpayOrderId: "order_test_" + Date.now(),
      razorpayPaymentId: "pay_test_" + Date.now(),
      status: "confirmed"
    });

    await testBooking.save();
    console.log("   ✅ Test booking created successfully!");
    console.log(`      Booking ID: ${testBooking._id}`);
    console.log(`      Vehicle: ${selectedVehicle.model}`);
    console.log(`      Price: ₹${testBooking.totalPrice}`);
    console.log(`      Pickup: ${selectedVehicle.district} - ${selectedVehicle.location}`);
    console.log(`      Date: ${pickupDate.toLocaleDateString()} to ${dropoffDate.toLocaleDateString()}`);

    // Step 4: Verify booking in database
    console.log("\n📋 STEP 4: Verifying Booking...");
    const verifyBooking = await Booking.findById(testBooking._id)
      .populate('user_id', 'username email')
      .populate('vehicle_id', 'model registeration_number');
    
    if (verifyBooking) {
      console.log("   ✅ Booking verified in database!");
      console.log(`      User: ${verifyBooking.user_id.username}`);
      console.log(`      Vehicle: ${verifyBooking.vehicle_id.model}`);
    }

    // Step 5: Check all bookings for user
    console.log("\n📋 STEP 5: User's All Bookings...");
    const userBookings = await Booking.find({ user_id: testUser._id });
    console.log(`   ✅ User has ${userBookings.length} booking(s)`);

    console.log("\n" + "=".repeat(50));
    console.log("🎉 ALL TESTS PASSED SUCCESSFULLY!");
    console.log("=".repeat(50));

    console.log("\n📊 SUMMARY:");
    console.log("   - Database: ✅ Connected");
    console.log("   - User: ✅ Found");
    console.log("   - Vehicles: ✅ Available");
    console.log("   - Booking: ✅ Created & Verified");

    console.log("\n💡 To test the full flow with Razorpay:");
    console.log("   1. Go to http://localhost:5173");
    console.log("   2. Login with test@gmail.com");
    console.log("   3. Search for a car and book it");
    console.log("   4. Use Razorpay test card: 4111 1111 1111 1111");

  } catch (error) {
    console.error("\n❌ TEST FAILED:", error.message);
    console.error(error);
  } finally {
    await mongoose.disconnect();
    console.log("\n✅ Disconnected from MongoDB");
  }
}

testBookingFlow();
